﻿#pragma once

#ifdef USE_APL_ARRAY

#include <algorithm>
#include <assert.h>
#include <string.h>

//////////////////////////////////////////////////////////////////////////
// массив, копи-паст CArray, т.к. CaplTAggr пользоваться не возможно,
// особенно обязательным определением операции приведения элемента (мы будем чаще всего использовать строки) к long
// функции-члены в основном повторяют MFC-шный CArray

template <class TYPE, class REF_TYPE = const TYPE&>
class CaplArray
{
protected:
	TYPE* m_pData;		// реальный массив
	int m_nSize;		// число элементов массива (upperBound - 1)
	int m_nMaxSize;		// выделенная память под массив
	int m_nGrowBy;		// число, на которое увеличивается размер массива

public:
	CaplArray();
	~CaplArray();

	// число элементов в массиве
	int GetCount() const { return m_nSize; }
	int GetSize() const { return m_nSize; }

	// текущий максимальный индекс в массиве (если массив пустой - вохвращает -1)
	int GetUpperBound() const { return m_nSize - 1; }

	// установить размер массива
	bool SetSize(int nNewSize, int nGrowBy = -1);

	// очистить массив
	void RemoveAll() { SetSize(0, -1); }

	// получить ссылку на элемент массива
	TYPE& ElementAt(int nIndex);
	const TYPE& ElementAt(int nIndex) const;

	// получить значение элемента массива
	TYPE& GetAt(int nIndex);
	const TYPE& GetAt(int nIndex) const;

	// прямой доступ к массиву
	const TYPE* GetData() const;
	TYPE* GetData();

	// установить элемент массива
	bool SetAt(int nIndex, REF_TYPE newElement);

	// установить элемент массива с расширением (если нужно) размера массива
	bool SetAtGrow(int nIndex, REF_TYPE newElement);

	// добавить элемент в массив
	int Add(REF_TYPE newElement);

	// добавить другой массив
	int Append(const CaplArray& src);

	// заменить элементы массива элементами другого массива
	void Copy(const CaplArray& src);

	// вставить элемент
	bool InsertAt(int nIndex, REF_TYPE newElement, int nCount = 1);
	bool InsertAt(int nStartIndex, CaplArray* pNewArray);

	// пустой ли массив
	bool IsEmpty() const { return m_nSize == 0; }

	// удалить элемент(ы)
	bool RemoveAt(int nIndex, int nCount = 1);

	// оператор прямого доступа к элементу
	TYPE& operator [](int nIndex);
	const TYPE& operator [](int nIndex) const;
};

template<class TYPE, class REF_TYPE>
CaplArray<TYPE, REF_TYPE>::CaplArray()
{
	m_pData = NULL;
	m_nSize = m_nMaxSize = m_nGrowBy = 0;
}

template<class TYPE, class REF_TYPE>
CaplArray<TYPE, REF_TYPE>::~CaplArray()
{
	if (m_pData != NULL)
	{
		for (int i = 0; i < m_nSize; ++i)
			(m_pData + i)->~TYPE();
		delete[](unsigned char*)m_pData;
	}
}

template<class TYPE, class REF_TYPE>
bool CaplArray<TYPE, REF_TYPE>::SetSize(int nNewSize, int nGrowBy /*= -1*/)
{
	if (nNewSize < 0)
	{
		return false;
	}

	if (nGrowBy >= 0)
		m_nGrowBy = nGrowBy;  // устанавливаем новый размер

	if (nNewSize == 0)
	{
		// удаляем массив
		if (m_pData != NULL)
		{
			for (int i = 0; i < m_nSize; i++)
				(m_pData + i)->~TYPE();
			delete[](unsigned char*)m_pData;
			m_pData = NULL;
		}
		m_nSize = m_nMaxSize = 0;
	}
	else if (m_pData == NULL)
	{
		// выделяем буфер для максимального значения из величин требуемого числа элементов массива или требуемого размера буфера
        //size_t nAllocSize = std::max(nNewSize, m_nGrowBy);
		size_t nAllocSize = nNewSize;
		if(m_nGrowBy>nNewSize) nAllocSize=m_nGrowBy;

		m_pData = (TYPE*) new unsigned char[(size_t)nAllocSize * sizeof(TYPE)];
		memset((void*)m_pData, 0, (size_t)nAllocSize * sizeof(TYPE));
		for (int i = 0; i < nNewSize; i++)
			::new((void*)(m_pData + i)) TYPE;
		m_nSize = nNewSize;
		m_nMaxSize = nAllocSize;
	}
	else if (nNewSize <= m_nMaxSize)
	{
		// если помещается в выделенную память
		if (nNewSize > m_nSize)
		{
			// инициализируем новые элементы массива (выделение памяти не требуется)
			memset((void*)(m_pData + m_nSize), 0, (size_t)(nNewSize - m_nSize) * sizeof(TYPE));
			for (int i = 0; i < nNewSize - m_nSize; i++)
				::new((void*)(m_pData + m_nSize + i)) TYPE;
		}
		else if (m_nSize > nNewSize)
		{
			// удаляем старые элементы
			for (int i = 0; i < m_nSize - nNewSize; ++i)
				(m_pData + nNewSize + i)->~TYPE();
		}
		m_nSize = nNewSize;
	}
	else
	{
		// нужно увеличить буфер
		nGrowBy = m_nGrowBy;
		if (nGrowBy == 0)
		{
			// если неопределена - задаем эвристически (чтобы не насиловать кучу)
			nGrowBy = m_nSize / 8;
			nGrowBy = (nGrowBy < 4) ? 4 : ((nGrowBy > 1024) ? 1024 : nGrowBy);
		}
		int nNewMax;
		if (nNewSize < m_nMaxSize + nGrowBy)
			nNewMax = m_nMaxSize + nGrowBy;  // granularity
		else
			nNewMax = nNewSize;  // no slush

        assert(nNewMax >= m_nMaxSize);  // no wrap around

		if (nNewMax < m_nMaxSize)
			return false;

		TYPE* pNewData = (TYPE*) new unsigned char[(size_t)nNewMax * sizeof(TYPE)];

		// копируем данные из старого буфера в новый
		memcpy((unsigned char*)pNewData, m_pData, (size_t)m_nSize * sizeof(TYPE));

		// инициализируем оставшиеся элементы
        assert(nNewSize > m_nSize);
		memset((void*)(pNewData + m_nSize), 0, (size_t)(nNewSize - m_nSize) * sizeof(TYPE));
		for (int i = 0; i < nNewSize - m_nSize; i++)
			::new((void*)(pNewData + m_nSize + i)) TYPE;

		// удаляем старый буфер (деструкторы элементов не вызываются)
		delete[](unsigned char*)m_pData;
		m_pData = pNewData;
		m_nSize = nNewSize;
		m_nMaxSize = nNewMax;
	}
	return true;
}

template<class TYPE, class REF_TYPE>
inline const TYPE& CaplArray<TYPE, REF_TYPE>::ElementAt(int nIndex) const
{
	if (nIndex >= 0 && nIndex < m_nSize)
	{
		return m_pData[nIndex];
	}
	static TYPE bad_value;
	return bad_value;
}

template<class TYPE, class REF_TYPE>
inline TYPE& CaplArray<TYPE, REF_TYPE>::ElementAt(int nIndex)
{
	if (nIndex >= 0 && nIndex < m_nSize)
	{
		return m_pData[nIndex];
	}

	static TYPE bad_value;
	return bad_value;
}

template<class TYPE, class REF_TYPE>
inline const TYPE& CaplArray<TYPE, REF_TYPE>::GetAt(int nIndex) const
{
	if (nIndex >= 0 && nIndex < m_nSize)
	{
		return m_pData[nIndex];
	}

	static TYPE bad_value;
	return bad_value;
}

template<class TYPE, class REF_TYPE>
inline TYPE& CaplArray<TYPE, REF_TYPE>::GetAt(int nIndex)
{
	if (nIndex >= 0 && nIndex < m_nSize)
	{
		return m_pData[nIndex];
	}
	static TYPE bad_value;
	return bad_value;
}

template<class TYPE, class REF_TYPE>
inline const TYPE* CaplArray<TYPE, REF_TYPE>::GetData() const
{
	return (const TYPE*)m_pData;
}

template<class TYPE, class REF_TYPE>
inline TYPE* CaplArray<TYPE, REF_TYPE>::GetData()
{
	return (TYPE*)m_pData;
}

template<class TYPE, class REF_TYPE>
inline TYPE& CaplArray<TYPE, REF_TYPE>::operator[](int nIndex)
{
	return ElementAt(nIndex);
}

template<class TYPE, class REF_TYPE>
inline const TYPE& CaplArray<TYPE, REF_TYPE>::operator[](int nIndex) const
{
	return GetAt(nIndex);
}

template<class TYPE, class REF_TYPE>
inline int CaplArray<TYPE, REF_TYPE>::Add(REF_TYPE newElement)
{
	int nIndex = m_nSize;
	SetAtGrow(nIndex, newElement);
	return nIndex;
}

template<class TYPE, class REF_TYPE>
bool CaplArray<TYPE, REF_TYPE>::SetAt(int nIndex, REF_TYPE newElement)
{
	if (nIndex >= 0 && nIndex < m_nSize)
	{
		m_pData[nIndex] = newElement;
		return true;
	}

	return false;
}

template<class TYPE, class REF_TYPE>
int CaplArray<TYPE, REF_TYPE>::Append(const CaplArray& src)
{
	if (this == &src)
		return 0;

	int nOldSize = m_nSize;
	SetSize(m_nSize + src.m_nSize);

	TYPE* pDest = m_pData + nOldSize;
	const TYPE* pSrc = src.m_pData;
	size_t nCount = src.m_nSize;
	while (nCount--) *pDest++ = *pSrc++;

	return nOldSize;
}

template<class TYPE, class REF_TYPE>
void CaplArray<TYPE, REF_TYPE>::Copy(const CaplArray& src)
{
	if (this != &src)
	{
		SetSize(src.m_nSize);
		TYPE* pDest = m_pData;
		const TYPE* pSrc = src.m_pData;
		size_t nCount = src.m_nSize;
		while (nCount--) *pDest++ = *pSrc++;
	}
}

template<class TYPE, class REF_TYPE>
bool CaplArray<TYPE, REF_TYPE>::SetAtGrow(int nIndex, REF_TYPE newElement)
{
	if (nIndex < 0)
	{
		return false;
	}

	if (nIndex >= m_nSize)
		SetSize(nIndex + 1, -1);
	m_pData[nIndex] = newElement;

	return true;
}

template<class TYPE, class REF_TYPE>
bool CaplArray<TYPE, REF_TYPE>::InsertAt(int nIndex, REF_TYPE newElement, int nCount /* = 1 */)
{
	if (nIndex < 0 || nCount <= 0)
	{
		return false;
	}

	if (nIndex >= m_nSize)
	{
		// добавляем в конец массива
		SetSize(nIndex + nCount, -1);
	}
	else
	{
		// вставляем в середину массива
		int nOldSize = m_nSize;
		SetSize(m_nSize + nCount, -1);  // увеличиваем размер массива

		// очищаем память перед копированием
		for (int i = 0; i < nCount; i++)
			(m_pData + nOldSize + i)->~TYPE();
		// смещаем старые данные в конец массива
#ifdef _MFC_VER
		memmove(m_pData + nIndex + nCount, (nOldSize - nIndex) * sizeof(TYPE), m_pData + nIndex, (nOldSize - nIndex) * sizeof(TYPE));
#else
		memmove((unsigned char*)(m_pData + nIndex + nCount), m_pData + nIndex, (nOldSize - nIndex) * (int)sizeof(TYPE));
#endif
		// инициализация
		memset((void*)(m_pData + nIndex), 0, (size_t)nCount * sizeof(TYPE));
		for (int i = 0; i < nCount; i++)
			::new((void*)(m_pData + nIndex + i)) TYPE;
	}

	// вставляем новое значение
    assert(nIndex + nCount <= m_nSize);
	while (nCount--)
		m_pData[nIndex++] = newElement;

	return true;
}

template<class TYPE, class REF_TYPE>
bool CaplArray<TYPE, REF_TYPE>::RemoveAt(int nIndex, int nCount /* = 1 */)
{
    assert(nIndex >= 0);
    assert(nCount >= 0);
	int nUpperBound = nIndex + nCount;
	assert(nUpperBound <= m_nSize);

	if (nIndex < 0 || nCount < 0 || (nUpperBound > m_nSize) || (nUpperBound < nIndex) || (nUpperBound < nCount))
		return false;

	// удаляем элементы
	int nMoveCount = m_nSize - (nUpperBound);
	for (int i = 0; i < nCount; ++i)
		(m_pData + nIndex + i)->~TYPE();
	if (nMoveCount)
	{
#ifdef _MFC_VER
		memmove( (void*)m_pData + nIndex, (size_t)nMoveCount * sizeof(TYPE), m_pData + nUpperBound, (size_t)nMoveCount * sizeof(TYPE));
#else
		memmove((unsigned char*)(m_pData + nIndex), m_pData + nUpperBound, (size_t)nMoveCount * sizeof(TYPE));
#endif
	}
	m_nSize -= nCount;

	return true;
}

template<class TYPE, class REF_TYPE>
bool CaplArray<TYPE, REF_TYPE>::InsertAt(int nStartIndex, CaplArray* pNewArray)
{
    assert(pNewArray != NULL);
    assert(nStartIndex >= 0);

	if (pNewArray == NULL || nStartIndex < 0)
		return false;

	if (pNewArray->GetSize() > 0)
	{
		InsertAt(nStartIndex, pNewArray->GetAt(0), pNewArray->GetSize());
		for (int i = 0; i < pNewArray->GetSize(); ++i)
			SetAt(nStartIndex + i, pNewArray->GetAt(i));
	}
	return true;
}

#endif
